"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Input } from "@/components/ui/input"
import { Badge } from "@/components/ui/badge"
import { ArrowLeft, ShoppingCart } from "lucide-react"
import { useCart } from "@/components/cart-context"
import { useRouter } from "next/navigation"
import Link from "next/link"

const batteryData = {
  "1": {
    id: "1",
    name: "ROCKET NX110-5",
    subtitle: "70AH/600CCA",
    model: "ROCKET",
    priceWithTrade: 440.0,
    priceWithoutTrade: 490.0,
    image: "/rocket-battery-black-automotive.jpg",
    inStock: true,
    specifications: {
      tipo: "NX110-5",
      placas: "13",
      amperios: "70 Ah",
      cca: "600 A",
      largo: "260 mm",
      ancho: "173 mm",
      alto: "222 mm",
      peso: "17,90 kg",
      polaridad: "(+  -)",
    },
  },
  "2": {
    id: "2",
    name: "VARTA BLUE DYNAMIC",
    subtitle: "60AH/540CCA",
    model: "VARTA",
    priceWithTrade: 320.0,
    priceWithoutTrade: 370.0,
    image: "/rocket-battery-black-automotive.jpg",
    inStock: true,
    specifications: {
      tipo: "E11",
      placas: "12",
      amperios: "60 Ah",
      cca: "540 A",
      largo: "242 mm",
      ancho: "175 mm",
      alto: "190 mm",
      peso: "15,30 kg",
      polaridad: "(-  +)",
    },
  },
}

export default function ProductPage({ params }: { params: { id: string } }) {
  const [quantity, setQuantity] = useState(1)
  const [dejandoBateria, setDejandoBateria] = useState("no") // Updated default value
  const { addItem } = useCart()
  const router = useRouter()

  const battery = batteryData[params.id as keyof typeof batteryData]

  if (!battery) {
    return <div>Producto no encontrado</div>
  }

  const getCurrentPrice = () => {
    if (dejandoBateria === "si") {
      return battery.priceWithTrade
    } else if (dejandoBateria === "no") {
      return battery.priceWithoutTrade
    }
    return battery.priceWithTrade // Default to trade-in price
  }

  const handleAddToCart = () => {
    const currentPrice = getCurrentPrice()
    for (let i = 0; i < quantity; i++) {
      addItem({
        id: battery.id,
        name: battery.name,
        price: currentPrice,
        image: battery.image,
        model: battery.model,
      })
    }
    router.push("/")
  }

  return (
    <div className="min-h-screen bg-background">
      <div className="container mx-auto px-4 py-8">
        <Link href="/" className="inline-flex items-center gap-2 text-primary hover:text-primary/80 mb-6">
          <ArrowLeft className="w-4 h-4" />
          Volver al catálogo
        </Link>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
          {/* Product Image */}
          <div className="space-y-4">
            <div className="relative bg-white rounded-lg p-8 border">
              <img
                src={battery.image || "/placeholder.svg"}
                alt={battery.name}
                className="w-full h-96 object-contain"
              />
              {battery.inStock && (
                <Badge className="absolute top-4 left-4 bg-accent text-accent-foreground">En Stock</Badge>
              )}
            </div>
          </div>

          {/* Product Details */}
          <div className="space-y-6">
            <div>
              <h1 className="text-3xl font-bold text-foreground mb-2">
                {battery.name} – {battery.subtitle}
              </h1>
              <div className="flex items-center gap-4 mb-4">
                <span className="text-2xl font-bold text-orange-500">
                  S/{battery.priceWithTrade.toFixed(2)} – S/{battery.priceWithoutTrade.toFixed(2)}
                </span>
              </div>
            </div>

            {/* Specifications */}
            <div className="space-y-3">
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Tipo:</span>
                <span>{battery.specifications.tipo}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Placas:</span>
                <span>{battery.specifications.placas}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Amperios (Ah):</span>
                <span>{battery.specifications.amperios}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ CCA (-18°C):</span>
                <span>{battery.specifications.cca}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Largo:</span>
                <span>{battery.specifications.largo}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Ancho:</span>
                <span>{battery.specifications.ancho}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Alto:</span>
                <span>{battery.specifications.alto}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Peso(kg):</span>
                <span>{battery.specifications.peso}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="font-semibold">▶ Polaridad</span>
                <span>{battery.specifications.polaridad}</span>
              </div>
            </div>

            {/* Purchase Options */}
            <div className="space-y-4 pt-6 border-t">
              <div>
                <label className="block text-sm font-medium mb-2">Dejando Batería</label>
                <Select value={dejandoBateria} onValueChange={setDejandoBateria}>
                  <SelectTrigger>
                    <SelectValue placeholder="Dejando batería" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="no">Elige una opción</SelectItem> {/* Updated value */}
                    <SelectItem value="si">Dejando batería</SelectItem>
                    <SelectItem value="no">Sin dejar batería</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {dejandoBateria && (
                <div className="text-2xl font-bold text-gray-800">S/{getCurrentPrice().toFixed(2)}</div>
              )}

              <div className="flex items-center gap-4">
                <div className="flex items-center border rounded-md">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setQuantity(Math.max(1, quantity - 1))}
                    className="px-3"
                  >
                    -
                  </Button>
                  <Input
                    type="number"
                    value={quantity}
                    onChange={(e) => setQuantity(Math.max(1, Number.parseInt(e.target.value) || 1))}
                    className="w-16 text-center border-0 focus-visible:ring-0"
                    min="1"
                  />
                  <Button variant="ghost" size="sm" onClick={() => setQuantity(quantity + 1)} className="px-3">
                    +
                  </Button>
                </div>

                <Button
                  onClick={handleAddToCart}
                  className="flex-1 bg-purple-500 hover:bg-purple-600 text-white"
                  size="lg"
                  disabled={!dejandoBateria}
                >
                  <ShoppingCart className="w-4 h-4 mr-2" />
                  Añadir al carrito
                </Button>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
